#ifndef V3D_TOOLS_BOUNDAABB_H_INCLUDED
#define V3D_TOOLS_BOUNDAABB_H_INCLUDED

#include "Vector.h"
#include "Triangle.h"


namespace V3D {

//////////////////////////////////////////////////////////////////////////////////////////
// Classe de base des BoundAabb

class BoundAabb
{
	typedef Vector3f      Vector3f;
	typedef Vector3D      Vector3D;
	typedef Vect3DArray   Vect3DArray;
	typedef TriangleArray TriangleArray;


public:
	inline BoundAabb() {}
	explicit inline BoundAabb(const Vector3D& vcMin, const Vector3D& vcMax) 
		: m_vcMin(vcMin), m_vcMax(vcMax) {}

	inline void SetMinMax( const Vector3D& vcMin, const Vector3D& vcMax)
		{ m_vcMin = vcMin; m_vcMax = vcMax;}

	void Build(const Vect3DArray& aPointCloud, const TriangleArray& aTriangles);
	void Build(const Vect3DArray& aPointCloud);

	void UpdateForVertexAdd( const Vector3D& vcAdd);
	inline void Merge(const BoundAabb& other);
	inline void Intersection(const BoundAabb& other);
	inline void Intersection(const Vector3D& vcMin, const Vector3D& vcMax);

	inline void Translate( const Vector3D& vcTransl);

	void GetVertices( Vector3D aObbVerts[8]) const;
	inline const Vector3D& GetMin() const {return m_vcMin;}
	inline const Vector3D& GetMax() const {return m_vcMax;}
	inline Vector3D GetCenter() const;
	inline Vector3D GetHalfSize() const;
	inline void GetCenterAndHalfSize(Vector3D& vcCenter, Vector3D& vcHalfSize) const;

	inline float GetVolume() const;

	inline bool IsPointInside(const Vector3D& pt) const;
	inline bool Intersects(const BoundAabb& other) const;
	inline bool Intersects(const Vector3D& vcMin, const Vector3D& vcMax) const;

	
	static BoundAabb Union(const BoundAabb& box1, const BoundAabb& box2); 

private:
	Vector3D m_vcMin;
	Vector3D m_vcMax;
};


inline float BoundAabb::GetVolume() const
{
	Vector3D vcSize = m_vcMax; vcSize -= m_vcMin;
	return float(vcSize.x * vcSize.y * vcSize.z);
}


inline void BoundAabb::GetCenterAndHalfSize(Vector3D& vcCenter, Vector3D& vcHalfSize) const
{
	vcCenter   = m_vcMax; 
	vcCenter   += m_vcMin; 
	vcCenter   *= .5f;
	
	vcHalfSize = m_vcMax;
	vcHalfSize -= m_vcMin;
	vcHalfSize *= .5f;
}


inline Vector3D BoundAabb::GetCenter() const
{
	return (m_vcMax + m_vcMin) * .5f; 
}

inline Vector3D BoundAabb::GetHalfSize() const
{
	Vector3D res;
	res.Set((m_vcMax - m_vcMin) * .5f); 
	return res;
}


inline bool BoundAabb::IsPointInside(const Vector3D& pt) const
{
	return  ( pt.x >= m_vcMin.x && pt.y >= m_vcMin.y && pt.z >= m_vcMin.z 
		   && pt.x <= m_vcMax.x && pt.y <= m_vcMin.y && pt.z <= m_vcMin.z );
}

inline bool BoundAabb::Intersects(const BoundAabb& other) const
{
	return !(  m_vcMax.x < other.m_vcMin.x
		    || m_vcMin.x > other.m_vcMax.x
		    || m_vcMax.y < other.m_vcMin.y
		    || m_vcMin.y > other.m_vcMax.y
		    || m_vcMax.z < other.m_vcMin.z
		    || m_vcMin.z > other.m_vcMax.z );
/*

	float fDiffCentersX = (m_vcMax.x + m_vcMin.x ) - (other.m_vcMax.x + other.m_vcMin.x);
	float fDiffCentersY = (m_vcMax.y + m_vcMin.y ) - (other.m_vcMax.y + other.m_vcMin.y);
	float fDiffCentersZ = (m_vcMax.z + m_vcMin.z ) - (other.m_vcMax.z + other.m_vcMin.z);
	
	float fSumSizesX    = (m_vcMax.x - m_vcMin.x ) + (other.m_vcMax.x - other.m_vcMin.x);
	float fSumSizesY    = (m_vcMax.y - m_vcMin.y ) + (other.m_vcMax.y - other.m_vcMin.y);
	float fSumSizesZ    = (m_vcMax.z - m_vcMin.z ) + (other.m_vcMax.z - other.m_vcMin.z);

	return (  (  ( Maths::Abs(fDiffCentersX) < fSumSizesX)
	          && ( Maths::Abs(fDiffCentersY) < fSumSizesY)
	          && ( Maths::Abs(fDiffCentersZ) < fSumSizesZ) ) );
//	return ( !(  ( Maths::Abs(fDiffCentersX) > fSumSizesX)
//	           | ( Maths::Abs(fDiffCentersY) > fSumSizesY)
//	           | ( Maths::Abs(fDiffCentersZ) > fSumSizesZ) ) );
//	return (  (  ( Maths::FastFloatGetSign( Maths::Abs(fDiffCentersX) - fSumSizesX) ) // 1 si abs - sumx < 0
//	           & ( Maths::FastFloatGetSign( Maths::Abs(fDiffCentersY) - fSumSizesY) ) // 1 si abs - sumy < 0
//	           & ( Maths::FastFloatGetSign( Maths::Abs(fDiffCentersZ) - fSumSizesZ) ) ) != 0);
*/
}

inline bool BoundAabb::Intersects(const Vector3D& vcMin, const Vector3D& vcMax) const
{
	return !(  m_vcMax.x < vcMin.x
		    || m_vcMin.x > vcMax.x
		    || m_vcMax.y < vcMin.y
		    || m_vcMin.y > vcMax.y
		    || m_vcMax.z < vcMin.z
		    || m_vcMin.z > vcMax.z );
/*
	float fDiffCentersX = (m_vcMax.x + m_vcMin.x ) - (vcMax.x + vcMin.x);
	float fDiffCentersY = (m_vcMax.y + m_vcMin.y ) - (vcMax.y + vcMin.y);
	float fDiffCentersZ = (m_vcMax.z + m_vcMin.z ) - (vcMax.z + vcMin.z);
	
	float fSumSizesX    = (m_vcMax.x - m_vcMin.x ) + (vcMax.x - vcMin.x);
	float fSumSizesY    = (m_vcMax.y - m_vcMin.y ) + (vcMax.y - vcMin.y);
	float fSumSizesZ    = (m_vcMax.z - m_vcMin.z ) + (vcMax.z - vcMin.z);

	return (  (  ( Maths::Abs(fDiffCentersX) < fSumSizesX)
	          && ( Maths::Abs(fDiffCentersY) < fSumSizesY)
	          && ( Maths::Abs(fDiffCentersZ) < fSumSizesZ) ) );
*/
}



inline void BoundAabb::Translate( const Vector3D& vcTransl)
{
	m_vcMin += vcTransl;
	m_vcMax += vcTransl;
}


inline void BoundAabb::Intersection(const BoundAabb& other)
{
	Intersection( other.m_vcMin, other.m_vcMax );
}


inline void BoundAabb::Intersection(const Vector3D& vcMin, const Vector3D& vcMax)
{
	if( m_vcMin.x < vcMin.x) m_vcMin.x = vcMin.x;
	if( m_vcMin.y < vcMin.y) m_vcMin.y = vcMin.y;
	if( m_vcMin.z < vcMin.z) m_vcMin.z = vcMin.z;

	if( m_vcMax.x > vcMax.x) m_vcMax.x = vcMax.x;
	if( m_vcMax.y > vcMax.y) m_vcMax.y = vcMax.y;
	if( m_vcMax.z > vcMax.z) m_vcMax.z = vcMax.z;
}





inline void BoundAabb::UpdateForVertexAdd( const Vector3D& vcAdd)
{
	// Peut etre exploiter :
	//	fmin(f1,f2) = .5f * ( (f1 + f2) - abs(f1 - f2) ) ?
	// Mais apres un test : fabs -> appel de fonction

/*
	m_vcMin.x = std::min( m_vcMin.x, vcAdd.x);
	m_vcMin.y = std::min( m_vcMin.y, vcAdd.y);
	m_vcMin.z = std::min( m_vcMin.z, vcAdd.z);

	m_vcMax.x = std::max( m_vcMax.x, vcAdd.x);
	m_vcMax.y = std::max( m_vcMax.y, vcAdd.y);
	m_vcMax.z = std::max( m_vcMax.z, vcAdd.z);
*/
/*
	m_vcMin.x = std::min( m_vcMin.x, vcAdd.x);
	m_vcMin.y = .5f * float((m_vcMin.y + vcAdd.y) - fabs( m_vcMin.y - vcAdd.y ) );
	m_vcMin.z = .5f * float((m_vcMin.z + vcAdd.z) - fabs( m_vcMin.z - vcAdd.z ) );

	m_vcMax.x = .5f * float((m_vcMax.x + vcAdd.x) - fabs( m_vcMax.x - vcAdd.x ) );
	m_vcMax.y = .5f * float((m_vcMax.y + vcAdd.y) - fabs( m_vcMax.y - vcAdd.y ) );
	m_vcMax.z = .5f * float((m_vcMax.z + vcAdd.z) - fabs( m_vcMax.z - vcAdd.z ) );
*/

	if( m_vcMin.x > vcAdd.x) m_vcMin.x = vcAdd.x;
	if( m_vcMin.y > vcAdd.y) m_vcMin.y = vcAdd.y;
	if( m_vcMin.z > vcAdd.z) m_vcMin.z = vcAdd.z;
	if( m_vcMax.x < vcAdd.x) m_vcMax.x = vcAdd.x;
	if( m_vcMax.y < vcAdd.y) m_vcMax.y = vcAdd.y;
	if( m_vcMax.z < vcAdd.z) m_vcMax.z = vcAdd.z;

}


inline void BoundAabb::Merge(const BoundAabb& other)
{
	if( m_vcMin.x > other.m_vcMin.x) m_vcMin.x = other.m_vcMin.x;
	if( m_vcMin.y > other.m_vcMin.y) m_vcMin.y = other.m_vcMin.y;
	if( m_vcMin.z > other.m_vcMin.z) m_vcMin.z = other.m_vcMin.z;
	if( m_vcMax.x < other.m_vcMax.x) m_vcMax.x = other.m_vcMax.x;
	if( m_vcMax.y < other.m_vcMax.y) m_vcMax.y = other.m_vcMax.y;
	if( m_vcMax.z < other.m_vcMax.z) m_vcMax.z = other.m_vcMax.z;
}










} // namespaces



#endif	// #ifndef V3D_TOOLS_BOUNDAABB_H_INCLUDED


